﻿
-- module setup
local me = { name = "trigger"}
local mod = thismod
mod[me.name] = me

--[[
Trigger.lua

A trigger is an important event that the entire raid group should be aware of. When one player detects the event, they broadcast a notification to the raid. Other raid members will accept the event after it has been confirmed by a second broadcast from someone else. One use is to allow people running an old version of the mod to accept threat resets and master target sets from other players if their boss mods are out of date.

------------------------------------------------------------------------
		Usage
------------------------------------------------------------------------

1) mytrigger = mod.trigger.createnew(name, timeout, cooldown, trigger)

	Trigger is a function to execute. The signature is

		trigger = function(value)

	<value> is an optional piece of data sent in the broadcast message. See the method comments for more details.

2) mytrigger:notify(author, value)

	see the method comments for more details.

------------------------------------------------------------------------
		Schema
------------------------------------------------------------------------

Each trigger has 
1) A timeout. This is how close together two broadcasts have to be in order to accept the event.
2) A cooldown. This is the minimum time btween two triggers. Events received on cooldown will be ignored.
3) A trigger function. This is a callback function run when the event is accepted.

We also record
4) The player who has sent the last broadcast
5) The value they sent (possibly nil). Won't accept a confirmation unless values match.
6) Whether the trigger is active (awaiting confirmation) or not (idle).
7) When the trigger was last activated (either triggered or broadcast)
]]

--[[
me.mytraces = 
{
	default = "info",
}
]]

me.instances = { }	-- list of triggers

--[[
mytrigger = mod.trigger.createnew(timeout, cooldown, trigger)

Trigger constructor.

<name>		string; identifier.
<timeout>	number; see (1) above
<cooldown>	number; see (2) above
<trigger>	function; see (3) above

Returns:
<mytrigger>	pointer to a Trigger instance.
]]
me.createnew = function(name, timeout, cooldown, trigger)
	
	local mytrigger = 
	{
		-- fixed values
		name = name,
		timeout = timeout,
		cooldown = cooldown,
		trigger = trigger,
		
		-- variables
		isactive = false,
		lastauthor = "",
		lasttime = 0,
		lastvalue = "",
		
		-- functions
		notify = me.notify,
		isoncooldown = me.isoncooldown,
		execute = me.execute,
	}

	table.insert(me.instances, mytrigger)
	return mytrigger 
	
end

me.myonupdates = 
{
	checkfortimeouts = 0.1,
}

me.checkfortimeouts = function()
	
	local timenow = GetTime()
	
	for index, value in pairs(me.instances) do
		if value.isactive and timenow > value.lasttime + value.timeout then
			
			-- timeout
			value.isactive = false
			
			if mod.trace.check("info", me, "timeout") then
				mod.trace.printf("The trigger %s was not confirmed and has timed out. It was activated by %s.", value.name, value.lastauthor)
			end
		end
	end
	
end

--[[
mytrigger:notify(author, value)

Call this method on a trigger instance when someone broadcasts it.

<author>		string; name of the player who broadcast the event.
<value		any; the value they sent, possible nil.
]]
me.notify = function(mytrigger, author, value)
	
	local timenow = GetTime()
	
	-- 1) Trigger is inactive
	if mytrigger.isactive == false then
		
		-- ignore if the trigger is on cooldown
		if timenow < mytrigger.lasttime + mytrigger.cooldown then
			return
		end
		
		-- enable
		mytrigger.isactive = true
		mytrigger.lastauthor = author
		mytrigger.lastvalue = value
		mytrigger.lasttime = timenow
		
		-- debug
		if mod.trace.check("info", me, "activate") then
			mod.trace.printf("The trigger %s is activated by %s with the value %s.", mytrigger.name, author, value)
		end
		
		-- if it comes from the user, doesn't need confirmation
		if author == UnitName("player") then
			mytrigger:execute(author)
		end
			
	-- 2) Trigger is active (awaiting confirmation)
	else
		
		-- ignore messages from the same player as the origina
		if author == mytrigger.lastauthor then
			return
		end
		
		-- check for value mismatch
		if value ~= mytrigger.lastvalue then
			
			if mod.trace.check("info", me, "valuemismatch") then
				mod.trace.printf("The trigger %s was given the value %s by %s, but %s gave the value %s.", mytrigger.name, tostring(mytrigger.lastvalue), author, tostring(value))
			end
			
			-- overwrite value and player
			mytrigger.lastvalue = value
			mytrigger.lasttime = timenow
			mytrigger.lastauthor = author
		
		-- value matches. This is a confirmation
		else
			mytrigger:execute(author)
		end
	end
	
end

--[[
mytrigger:execute(confirmer)

Called when the trigger is confirmed and runs.
]]
me.execute = function(mytrigger, confirmer)
	
	if mod.trace.check("info", me, "execute") then
		mod.trace.printf("The trigger %s is executing with value %s. It was activated by %s and confirmed by %s.", mytrigger.name, mytrigger.lastvalue, mytrigger.lastauthor, confirmer)
	end
	
	-- reset
	mytrigger.isactive = false
	mytrigger.lasttime = GetTime()
	
	-- run
	mytrigger.trigger(mytrigger.lastvalue)
		
end

--[[
bool = mytrigger:isoncooldown()

Returns true if the trigger is on cooldown. That is, it has procced recently. Note that if the trigger is active, it can't be on cooldown.
]]
me.isoncooldown = function(mytrigger)
	
	return (mytrigger.isactive == false) and (GetTime() < mytrigger.lasttime + mytrigger.cooldown)
	
end