﻿
-- module setup
local me = { name = "special"}
local mod = thismod
mod[me.name] = me

--[[
SpecialAbility.lua (2007 / 04 / 29)

This module detects abilities that don't cause damage but still cause significant threat. Since they don't cause damage, they often aren't detectable from combat log events alone. Examples are Sunder Armor, Faerie Fire, Counterspell, + random debuff applications like warlock curses. The types of abilities can be divided into physical ones, which can be parried, dodged, etc, and spells, which can be resisted.


--------------------------------------------------------
		World of Warcraft Implementation
--------------------------------------------------------

When we attempt to cast a spell, a UNIT_SPELLCAST_SENT event is raised; the spell name is arg2 and the  name of the target is arg4. When the spell actually fires (no out of range or similar errors), a UNIT_SPELLCAST_SUCCEEDED event is raised; the spell name is arg2.

When an ability is parried, dodged, misses, or the target is immune, we receive a CHAT_MSG_SPELL_SELF_DAMAGE event with format strings SPELLPARRIEDSELFOTHER, SPELLDODGEDSELFOTHER, SPELLMISSEDSELFOTHER, and IMMUNESPELLSELFOTHER respectively. For spells, there is also SPELLRESISTSELFOTHER and SPELLIMMUNESELFOTHER, for resists and the mob being immune to a mechanic (e.g. silence immune boss).


---------------------------------------------------------
		KTM Implementation
---------------------------------------------------------

We keep a list of all the spells and abilities tracked in this module in <me.spells>. The list <me.namelookup> converts a localised name such as "Sunder Armor" to its index in <me.spells>, 1 in this case.

When we receive UNIT_SPELLCAST_SENT event matching the spell we record the target in <me.spells[i].target>. When we receive a UNIT_SPELLCAST_SUCCESSFUL event matching the spell we generate a standard attack in the <combat> module. If we detect a miss / parry / etc, we generate an attack in the <combat> module with -1 hits to deduct the threat.

Currently we are working out the threat of the ability ourselves, because Combat.lua doesn't have a good mechanism to subtract threat, which we need to do. We run the risk of not taking into account special class or set threat bonuses, but for the current abilities there are no problems.


--------------------------------------------------------
		Interaction With Other Modules
--------------------------------------------------------

- We use the method <mod.combat.lognormalevent> to add or subtract threat from our abilities. 
- For each ability in <me.spells>, there must be a key in <mod.data.spells> matching the <id> value.

- The module requires these localisation keys to work perfectly (english examples)
"spell" - "sunder"		=	"Sunder Armor"
"spell" - "faeriefire"	=	"Faerie Fire (Feral)"


--------------------------------------------------------
				Debugging
--------------------------------------------------------

To enable all debugging of this module, uncomment these lines:

	me.mytraces = 
	{
		default = "info",
	}

The following trace prints exist:

"send"	-	UNIT_SPELLCAST_SENT
"fail"	-	parry / dodge / resist etc
"cast"	-	UNIT_SPELLCAST_SUCCEEDED
"threat"	-	change in threat

For example, to show only "send" traces, uncomment these lines:

	me.mytraces = 
	{
		send = "info",
	}
]]

--[[
------------------------------------------------------------------------
					Member Variables
------------------------------------------------------------------------
]]

--[[
At runtime, each subtable becomes e.g.
{
	id = "sunder", 			-- localisation key
	type = "physical",		-- either "physical" or "spell"
	name = "Sunder Armor"	-- localised name
	target = "..."				-- name of the last targetted mob
}
]]
me.spells = 
{
	{
		id = "sunder",
		type = "physical",
	}, 
	{
		id = "faeriefire",
		type = "spell",
	},
}

me.namelookup = { } --[[ Here's an example using the enUS localisation. The key is the localised string. The value is the index in <me.spell> that refers to the same spell.

me.namelookup = 
{
	["Sunder Armor"] = 1,
	["Faerie Fire"] = 2,
}
]]

--[[
-----------------------------------------------------------------------------
					Services from Loader.lua
-----------------------------------------------------------------------------
]]

--[[
When the module loads, we check that all the defined spells have a localisation string. If there is no value for the current locale, that spell won't be loaded or checked for.
]]
me.onload = function()
	
	local spellname
	
	-- check for localisation values
	for index, value in ipairs(me.spells) do
		spellname = mod.string.tryget("spell", value.id)
		
		if spellname then
			me.namelookup[spellname] = index
			
			value.name = spellname
			value.target = ""
		end
	end
	
end

--[[
---------------------------------------------------------------------------
					Services from Regex.lua
---------------------------------------------------------------------------
]]

me.myparsers = 
{
	{"parry", "SPELLPARRIEDSELFOTHER", "CHAT_MSG_SPELL_SELF_DAMAGE"}, -- Your %s was parried by %s.
	{"miss", "SPELLMISSSELFOTHER", "CHAT_MSG_SPELL_SELF_DAMAGE"},	-- Your %s missed %s.
	{"dodge", "SPELLDODGEDSELFOTHER", "CHAT_MSG_SPELL_SELF_DAMAGE"},	-- Your %s was dodged by %s.
	{"immune", "IMMUNESPELLSELFOTHER", "CHAT_MSG_SPELL_SELF_DAMAGE"},	-- %s is immune to your %s.
	{"resist", "SPELLRESISTSELFOTHER", "CHAT_MSG_SPELL_SELF_DAMAGE"}, -- Your %s was resisted by %s.
	{"immune2", "SPELLIMMUNESELFOTHER", "CHAT_MSG_SPELL_SELF_DAMAGE"}, -- Your %s failed. %s is immune.
}

me.onparse = function(identifier, spell, target)

	-- the "immune" parser has the spell and target the other way round, so swap it here
	if identifier == "immune" then
		local temp = spell
		spell = target
		target = temp
	end

	-- Check for the spell being in our database
	local spellindex = me.namelookup[spell]
	
	if spellindex == nil then
		return
	end
	
	-- debug
	if mod.trace.check("info", me, "fail") then
		mod.trace.printf("%s failed on %s, result was %s.", spell, target, identifier)
	end
	
	-- retract
	me.addspellthreat(spellindex, -1)
	
end

--[[
-----------------------------------------------------------------------------
				Services from Events.lua
-----------------------------------------------------------------------------
]]

me.myevents = { "UNIT_SPELLCAST_SUCCEEDED", "UNIT_SPELLCAST_SENT" }

me.onevent = function()

	-- check for spell match (arg2 for both events)
	local spellindex = me.namelookup[arg2]
	
	if spellindex == nil then
		return
	end

	if event == "UNIT_SPELLCAST_SENT" then
		me.spells[spellindex].target = arg4
		
		-- debug
		if mod.trace.check("info", me, "send") then
			mod.trace.printf("sending %s against %s.", arg2, arg4)
		end
	
	elseif event == "UNIT_SPELLCAST_SUCCEEDED" then
		
		-- debug
		if mod.trace.check("info", me, "cast") then
			mod.trace.printf("%s has successfully cast.", me.spells[spellindex].name)
		end
		
		-- submit
		me.addspellthreat(spellindex, 1)
	end

end	

----------------------------------------------------------------

--[[
me.addspellthreat(spellindex, count)

Calculates the threat from an attack, and adds it to our threat.

<spellindex>	integer; index of <me.spells> matching the spell used.
<count>			integer; 1 for a cast, -1 for a retraction.
]]
me.addspellthreat = function(spellindex, count)
	
	local spelldata = me.spells[spellindex]
	
	-- check for master target match
	if mod.target.targetismaster(spelldata.target) == nil then
		return
	end

	-- 2) total threat
	local threat = mod.my.ability(spelldata.id, "threat") * mod.my.globalthreat.value * count
	
	if mod.trace.check("info", me, "threat") then
		mod.trace.printf("Adding %d threat from %s.", threat, spelldata.name)
	end
	
	-- 3) submit
	mod.combat.lognormalevent(spelldata.name, count, 0, threat)
	
end